<?php
/**
 * WP Scan Fix – Permission / Malware Inspector (EXEC)
 * 仕様:
 * - URL秘密キー必須: ?key=xxxxx がないとアクセス不可
 * - 表示 -> チェック -> 確認 -> 削除 の2段階
 * - CSRFあり
 * - 表示カテゴリはセクション（タブではない）
 * - uploads はスキップ（走査しない）ただし .htaccess だけ例外で走査可能
 *
 * 表示カテゴリ:
 * ① パーミッション 555 のディレクトリ（ファイルもリストアップ・削除対象）
 * ② パーミッション許可なし（忠告のみ）: 2階層までのディレクトリ単位 + 配下の異常ファイル数のみ
 * ③ 改ざんPHP（削除対象）
 * ④ マルウェアPHP（削除対象）: 確定級2パターンのみで判定（誤検知を避ける）
 * ⑤ 不正 .htaccess（削除対象）: 高リスク中心 + あなたの見本パターンを検知
 *
 * 注意:
 * - このツールは非常に強力です。作業後は必ず削除してください。
 */

declare(strict_types=1);

ini_set('log_errors', '1');
ini_set('error_log', __DIR__ . '/wp-scan-fix-error.log');
error_reporting(E_ALL);
ini_set('display_errors', '0');


// --- セッション保存先の退避（サーバ設定が壊れている場合の保険） ---
$sessionDir = __DIR__ . '/.wp-scan-fix-session';
if (!is_dir($sessionDir)) {
    @mkdir($sessionDir, 0700, true);
}
if (is_dir($sessionDir) && is_writable($sessionDir)) {
    @ini_set('session.save_path', $sessionDir);
}
// ----------------------------------------------------------

error_reporting(E_ALL & ~E_WARNING);
ini_set('display_errors', '0');
ini_set('memory_limit', '256M');
set_time_limit(0);

mb_internal_encoding('UTF-8');

// セッションが動かないとCSRFは成立しないので、握りつぶさずに止める
if (session_status() !== PHP_SESSION_ACTIVE) {
    $https = (!empty($_SERVER['HTTPS']) && strtolower((string)$_SERVER['HTTPS']) !== 'off');

    if (PHP_VERSION_ID >= 70300) {
        session_set_cookie_params([
            'lifetime' => 0,
            'path' => '/',
            'secure' => $https,
            'httponly' => true,
            'samesite' => 'Lax',
        ]);
    } else {
        session_set_cookie_params(0, '/', '', $https, true);
    }

    @session_name('wp_scan_fix');

    if (!session_start()) {
        header('Content-Type: text/html; charset=UTF-8');
        echo '<h2>セッション開始に失敗しました</h2>';
        echo '<p>CSRFが使えないため処理できません。session.save_path の書き込み権限、Cookie、リダイレクト設定を確認してください。</p>';
        exit;
    }
}

/* =============================
 * 設定
 * ============================= */
const TOOL_VERSION = 'exec-2.1.0';

const SECRET_KEY = 'xxxxxxxxxxxxxxxxxxxxxxx'; // 必ず変更してください
const MAX_BUCKET_DEPTH = 2; // ディレクトリ単位の「全選択」用の分類（2階層）
const MAX_FILE_DEPTH_IN_555 = 2; // ①の555配下ファイル列挙深さ

const RECOMMENDED_DIR_PERM = '755';
const RECOMMENDED_FILE_PERM = '644';

const UPLOADS_REL = 'wp-content/uploads';

// public_html 外スキャン用
const MAX_OUTSIDE_DEPTH = 2;

/* 直下のWP標準PHP */
$WP_ROOT_STANDARD_PHP = [
    'index.php',
    'wp-activate.php',
    'wp-blog-header.php',
    'wp-comments-post.php',
    'wp-config.php',
    'wp-config-sample.php',
    'wp-cron.php',
    'wp-links-opml.php',
    'wp-load.php',
    'wp-login.php',
    'wp-mail.php',
    'wp-settings.php',
    'wp-signup.php',
    'wp-trackback.php',
    'xmlrpc.php',
];

/* 明らかに怪しいファイル名（必要なら増やす） */
$SUSPECT_NAMES_EXACT = [
    'wp-confiq.php',
];

/* =============================
 * 共通ユーティリティ
 * ============================= */
function h(string $s): string {
    return htmlspecialchars($s, ENT_QUOTES | ENT_SUBSTITUTE, 'UTF-8');
}

function norm(string $p): string {
    $p = str_replace('\\', '/', $p);
    $p = preg_replace('#/+#', '/', $p) ?? $p;
    return $p;
}

function rel_path(string $abs, string $root): string {
    $absN = norm($abs);
    $rootN = rtrim(norm($root), '/');
    if (strpos($absN, $rootN) === 0) {
        $r = substr($absN, strlen($rootN));
        if ($r === false) return $absN;
        $r = ltrim($r, '/');
        return $r === '' ? '.' : $r;
    }
    return $absN;
}

function file_perm_octal(string $path): string {
    $p = @fileperms($path);
    if ($p === false) return '-';
    return substr(sprintf('%o', $p), -3);
}

function is_within_root(string $path, string $root): bool {
    $rp = realpath($path);
    $rr = realpath($root);
    if ($rp === false || $rr === false) return false;
    $rpN = norm($rp);
    $rrN = rtrim(norm($rr), '/') . '/';
    return strpos($rpN . '/', $rrN) === 0;
}

function read_file_head(string $path, int $maxBytes): string {
    if (!is_file($path) || !is_readable($path)) return '';
    $fh = @fopen($path, 'rb');
    if (!$fh) return '';
    $data = @fread($fh, $maxBytes);
    @fclose($fh);
    if ($data === false) return '';
    return $data;
}

function parse_bool($v): bool {
    if (is_bool($v)) return $v;
    $s = strtolower((string)$v);
    return in_array($s, ['1','true','on','yes'], true);
}

function make_csrf_token(): string {
    if (empty($_SESSION['wp_scan_fix_csrf'])) {
        $_SESSION['wp_scan_fix_csrf'] = bin2hex(random_bytes(32));
    }
    return (string)$_SESSION['wp_scan_fix_csrf'];
}

function verify_csrf_token(?string $token): bool {
    return isset($_SESSION['wp_scan_fix_csrf'])
        && is_string($token)
        && hash_equals((string)$_SESSION['wp_scan_fix_csrf'], $token);
}

function key_ok(): bool {
    if (!isset($_GET['key'])) return false;
    $k = (string)$_GET['key'];
    return hash_equals(SECRET_KEY, $k);
}

function current_key(): string {
    return isset($_GET['key']) ? (string)$_GET['key'] : '';
}

function bucket_key(string $rel, int $depth): string {
    if ($rel === '.' || $rel === '') return '.';
    $parts = array_values(array_filter(explode('/', $rel), fn($x) => $x !== ''));
    $parts = array_slice($parts, 0, $depth);
    return $parts ? implode('/', $parts) : '.';
}

function depth_of_rel(string $rel): int {
    if ($rel === '.' || $rel === '') return 0;
    return substr_count(trim($rel, '/'), '/') + 1;
}

/* =============================
 * 削除/権限補助
 * ============================= */
function chmod_recursive(string $path, int $dirMode, int $fileMode, int $maxDepth = 25, int $depth = 0): bool {
    if ($depth > $maxDepth) return false;

    if (is_link($path)) {
        return true;
    }

    if (is_dir($path)) {
        @chmod($path, $dirMode);
        $items = @scandir($path);
        if ($items === false) return false;
        foreach ($items as $it) {
            if ($it === '.' || $it === '..') continue;
            $p = $path . DIRECTORY_SEPARATOR . $it;
            if (!chmod_recursive($p, $dirMode, $fileMode, $maxDepth, $depth + 1)) return false;
        }
        return true;
    }

    if (is_file($path)) {
        return @chmod($path, $fileMode);
    }

    return true;
}

function delete_path_recursive(string $path): bool {
    if (is_link($path) || is_file($path)) {
        return @unlink($path);
    }
    if (is_dir($path)) {
        $items = @scandir($path);
        if ($items === false) return false;
        foreach ($items as $it) {
            if ($it === '.' || $it === '..') continue;
            $p = $path . DIRECTORY_SEPARATOR . $it;
            if (!delete_path_recursive($p)) return false;
        }
        return @rmdir($path);
    }
    return false;
}


/* =============================
 * ③ 改ざんPHP判定（削除/隔離対象）
 * 目的:
 * - eval + 長い base64 + 復号/難読化の組み合わせを「改ざん」として検知
 * - 誤検知を減らすためスコア方式
 *
 * 注意:
 * - wp-includes / wp-admin / ルートのコアファイルは「削除」するとサイトが落ちるため、
 *   できれば「警告のみ」or「隔離+正規ファイルで復旧」の運用にする
 * ============================= */
function wpscanfix_is_tampered_php($content, &$reason = '') {
    $reason = '';
    $score = 0;
    $hits = [];

    $lc = strtolower($content);

    $exec_patterns = [
        'eval(',
        'assert(',
        'create_function(',
        'preg_replace(',
        'system(',
        'shell_exec(',
        'passthru(',
        'exec(',
        'popen(',
        'proc_open(',
    ];

    foreach ($exec_patterns as $p) {
        if (strpos($lc, $p) !== false) {
            $score += 4;
            $hits[] = "exec:$p";
        }
    }

    if (preg_match('~preg_replace\s*\(\s*[\'"][^\'"]*[eE][^\'"]*[\'"]\s*,~', $content)) {
        $score += 6;
        $hits[] = 'exec:preg_replace_e';
    }

    $obf_patterns = [
        'base64_decode',
        'gzinflate',
        'gzuncompress',
        'str_rot13',
        'rawurldecode',
        'urldecode(',
        'chr(',
        'goto ',
    ];

    foreach ($obf_patterns as $p) {
        if (strpos($lc, $p) !== false) {
            $score += 2;
            $hits[] = "obf:$p";
        }
    }

    if (preg_match('~ini_set\s*\(\s*[\'"]memory_limit[\'"]\s*,\s*[\'"]-1[\'"]\s*\)~i', $content)) {
        $score += 2;
        $hits[] = 'env:memory_limit_-1';
    }
    if (preg_match('~error_reporting\s*\(\s*0\s*\)~i', $content) || preg_match('~@error_reporting\s*\(\s*0\s*\)~i', $content)) {
        $score += 2;
        $hits[] = 'env:error_reporting_0';
    }
    if (preg_match('~set_time_limit\s*\(\s*(0|[1-9]\d{2,})\s*\)~i', $content)) {
        $score += 1;
        $hits[] = 'env:set_time_limit';
    }

    $base64_blob_count = 0;
    $max_blob_len = 0;

    if (preg_match_all('~[\'"]([A-Za-z0-9+/]{300,}={0,2})[\'"]~', $content, $m)) {
        foreach ($m[1] as $blob) {
            $base64_blob_count++;
            $len = strlen($blob);
            if ($len > $max_blob_len) $max_blob_len = $len;
        }
    }

    if ($base64_blob_count > 0) {
        $score += 4;
        $hits[] = 'obf:long_base64_blob';
    }
    if ($max_blob_len >= 1500) {
        $score += 2;
        $hits[] = 'obf:very_long_base64_blob';
    }

    if (preg_match('~PD9waHA~', $content)) {
        $score += 2;
        $hits[] = 'obf:base64_starts_php';
    }

    $request_patterns = [
        '$_get',
        '$_post',
        '$_request',
        '$_cookie',
        '$_server',
    ];
    foreach ($request_patterns as $p) {
        if (strpos($lc, $p) !== false) {
            $score += 1;
            $hits[] = "io:$p";
            break;
        }
    }

    if ($score >= 10) {
        $reason = implode(', ', array_unique($hits)) . " (score=$score)";
        return true;
    }

    return false;
}



/* =============================
 * ④ マルウェア判定（確定級 2パターン）
 * ============================= */
function is_definitely_malware_gecko_shell(string $filePath, int $maxReadBytes = 450000): array
{
    $res = [
        'is_malware' => false,
        'score' => 0,
        'matched' => [],
        'reasons' => [],
    ];

    if (!is_file($filePath) || !is_readable($filePath)) {
        $res['reasons'][] = 'ファイルを読めません';
        return $res;
    }

    $content = read_file_head($filePath, $maxReadBytes);
    if ($content === '') {
        $res['reasons'][] = '内容を取得できません';
        return $res;
    }

    $lc = strtolower($content);
    $score = 0;
    $matched = [];

    $indicators = [
        'title_gecko' => (strpos($lc, 'gecko [') !== false) || (strpos($lc, 'gecko [ <?=') !== false),
        'github_gecko' => (strpos($lc, 'github.com/madexploits/gecko') !== false),
        'backdoor_destroyer' => (strpos($lc, 'backdoor destroyer') !== false),
        'localroot_suggester' => (strpos($lc, 'localroot suggester') !== false),
        'exploit_db' => (strpos($lc, 'exploit-db.com') !== false),
        'pwnkit' => (strpos($lc, 'pwnkit') !== false),
        'terminal' => (strpos($lc, 'terminal') !== false) && (strpos($lc, 'submit-terminal') !== false),
        'upload' => (strpos($lc, 'move_uploaded_file') !== false) || (strpos($lc, 'enctype=') !== false && strpos($lc, 'multipart/form-data') !== false),
        'hex_unhex' => (strpos($lc, 'function unhex') !== false) && (preg_match('/\$\s*array\s*=\s*\[\s*[\s\S]{0,300}7368656c6c5f65786563/i', $content) === 1),
        'obf_array' => (preg_match('/\$\s*array\s*=\s*\[[\s\S]{0,1200}pwnkit/i', $content) === 1),
        'tool_menu' => (strpos($lc, 'tool-menu') !== false) && (strpos($lc, 'auto root') !== false),
    ];

    if ($indicators['title_gecko']) { $score += 6; $matched[] = 'Geckoタイトル'; }
    if ($indicators['github_gecko']) { $score += 8; $matched[] = 'MadExploits/Gecko'; }
    if ($indicators['backdoor_destroyer']) { $score += 8; $matched[] = 'Backdoor Destroyer'; }
    if ($indicators['localroot_suggester']) { $score += 4; $matched[] = 'Localroot Suggester'; }
    if ($indicators['exploit_db']) { $score += 4; $matched[] = 'exploit-db'; }
    if ($indicators['pwnkit']) { $score += 6; $matched[] = 'pwnkit'; }
    if ($indicators['terminal']) { $score += 4; $matched[] = 'terminal機能'; }
    if ($indicators['upload']) { $score += 4; $matched[] = 'upload機能'; }
    if ($indicators['hex_unhex']) { $score += 6; $matched[] = 'hex/unhex難読化'; }
    if ($indicators['tool_menu']) { $score += 2; $matched[] = 'ツールメニュー構造'; }
    if ($indicators['obf_array']) { $score += 2; $matched[] = '難読化配列'; }

    $hasCore = ($indicators['github_gecko'] || $indicators['title_gecko']) && $indicators['backdoor_destroyer'];
    $hasDangerTools = $indicators['terminal'] || $indicators['upload'] || $indicators['pwnkit'];

    if (($hasCore && $hasDangerTools) || $score >= 24) {
        $res['is_malware'] = true;
        $res['reasons'][] = 'Gecko系WebShellの固有特徴（配布元/表示/機能構造）が揃っています。';
        if ($indicators['pwnkit']) $res['reasons'][] = 'pwnkit等のローカルルート関連の処理が含まれています。';
        if ($indicators['terminal']) $res['reasons'][] = 'コマンド実行（ターミナル）機能が含まれています。';
        if ($indicators['upload']) $res['reasons'][] = 'アップロード機能が含まれています。';
    } else {
        $res['is_malware'] = false;
        $res['reasons'][] = '確定級の組み合わせに満たないため未確定扱いです。';
    }

    $res['score'] = $score;
    $res['matched'] = array_values(array_unique($matched));
    return $res;
}





/* =============================
 * ⑤ .htaccess 不正判定
 * ============================= */
/**
 * - WordPress標準のルート .htaccess（BEGIN/END WordPress）は除外
 * - 高リスク: 外部リダイレクト、php実行許可（AddHandler/SetHandler/AddType/auto_prepend_file 等）
 * - 中リスク: あなたの見本のような FilesMatch + Deny from all（大量の拡張子・混在ケース等）
 */
function is_suspicious_htaccess(string $filePath, string $relPath, string $rootAbs, int $maxReadBytes = 120000): array
{
    $res = [
        'is_bad' => false,
        'severity' => 'low', // low|medium|high
        'score' => 0,
        'kind' => '',
        'matched' => [],
        'reasons' => [],
    ];

    if (!is_file($filePath) || !is_readable($filePath)) {
        $res['reasons'][] = 'ファイルを読めません';
        return $res;
    }

    $content = read_file_head($filePath, $maxReadBytes);
    if ($content === '') {
        $res['reasons'][] = '内容を取得できません';
        return $res;
    }

    $lc = strtolower($content);
    $score = 0;
    $matched = [];

    // ルート直下の WordPress 標準 .htaccess は除外
    $isRootLevel = (strpos($relPath, '/') === false);
    if ($isRootLevel) {
        if (strpos($content, '# BEGIN WordPress') !== false && strpos($content, '# END WordPress') !== false) {
            $res['is_bad'] = false;
            $res['severity'] = 'low';
            $res['kind'] = 'wp_standard_root';
            $res['reasons'][] = 'WordPress標準のルート .htaccess の可能性が高いため除外しました。';
            return $res;
        }
    }

    // 高リスク1: 外部リダイレクト（RewriteRule/Redirect/ErrorDocument等）
    $hasRewrite = (strpos($lc, 'rewriterule') !== false) || (strpos($lc, 'redirect') !== false) || (strpos($lc, 'errordocument') !== false);
    $hasExternalUrl = (preg_match('#https?://[a-z0-9\.\-]+\.[a-z]{2,}#i', $content) === 1);
    $hasRewriteToExternal = $hasRewrite && $hasExternalUrl;
    if ($hasRewriteToExternal) {
        $score += 12;
        $matched[] = '外部URLリダイレクト/参照';
    }


    // 高リスク2: PHP実行許可/強制（uploadsや画像拡張子等をPHPとして扱う）
    $phpEnable = false;

    /**
     * PHPを「有効化」するルールのみ
     * （無効化はここに含めない）
     */
    $phpEnableRules = [
        // PHPハンドラを明示的に有効化
        '/\b(addhandler|sethandler)\b[\s\S]{0,200}(application\/x-httpd-php|php-script|cgi-script)/i',

        // PHPとして扱う拡張子を追加
        '/\baddtype\b[\s\S]{0,200}\bphp\b/i',

        // PHPエンジン明示ON
        '/php_flag\s+engine\s+on\b/i',

        // 自動プリロード系（即アウト）
        '/php_value\s+auto_(prepend|append)_file\b/i',

        // CGI 実行許可
        '/options\s+\+execcgi\b/i',
    ];

    /**
     * PHPを「無効化」するルール
     * （これがあれば有効化扱いしない）
     */
    $phpDisableRules = [
        '/\bsethandler\s+none\b/i',
        '/\bremovehandler\b.*\.php\b/i',
        '/php_flag\s+engine\s+off\b/i',
    ];

    // まず有効化を探す
    foreach ($phpEnableRules as $re) {
        if (preg_match($re, $content) === 1) {
            $phpEnable = true;
            break;
        }
    }

    // 無効化があれば必ず打ち消す
    if ($phpEnable) {
        foreach ($phpDisableRules as $re) {
            if (preg_match($re, $content) === 1) {
                $phpEnable = false;
                break;
            }
        }
    }

    // スコア反映
    if ($phpEnable) {
        $score += 14;
        $matched[] = 'PHP実行許可/強制';
    }

    // マルウェア型の allow 例外のみを検知（単一PHPをAllowしているか）
    $hasMalwareAllowException = false;

    if ($hasFilesMatch) {
        if (
            preg_match(
                '/<filesmatch[\s\S]{0,300}\^\(?[a-z0-9]{5,}\.php\)?\$[\s\S]{0,300}<\/filesmatch>/i',
                $content
            )
            && (
                strpos($lc, 'allow from all') !== false
                || strpos($lc, 'require all granted') !== false
            )
        ) {
            $hasMalwareAllowException = true;
        }
    }


    // 中リスク: FilesMatch + Deny from all（見本パターン含む）
    $hasFilesMatch = (strpos($lc, '<filesmatch') !== false);
    $hasDenyAll = (strpos($lc, 'deny from all') !== false) || (strpos($lc, 'require all denied') !== false);
    $hasOrderAllowDeny = (strpos($lc, 'order allow,deny') !== false) || (strpos($lc, 'order deny,allow') !== false);

    
    // 大量拡張子 or 大文字小文字の混在列（PHP|Php|PHp...）を検知
    $hasPhpVariants = (preg_match('/\bphp\|php5\|php7\|php8\b/i', $content) === 1) || (preg_match('/\bphp\|php\b/i', $content) === 1);
    $hasMixedCaseList = (preg_match('/\bphp\|php\b/i', $content) !== 1) && (preg_match('/\bPHP\|Php\|PHp\|pHp\|pHP\|phP\|PhP\b/', $content) === 1);
    $hasSuspectedExt = (strpos($lc, 'suspected') !== false);

    // 見本に近い: FilesMatch 内に "py|exe|phtml|php|...|suspected" のような列
    $looksLikeSample = false;
    if ($hasFilesMatch && $hasDenyAll) {
        if (preg_match('/<filesmatch[\s\S]{0,500}(py\|exe\|phtml\|php|phtml\|php|php\|php5|php5\|php7|php7\|php8)[\s\S]{0,500}<\/filesmatch>/i', $content) === 1) {
            $looksLikeSample = true;
        }
    }

    // deny + allow 例外が同時にある場合のみ中～高リスクとする
    if ($hasFilesMatch && $hasDenyAll && $hasMalwareAllowException) {
        $score += 10;
        $matched[] = 'deny + allow 例外';

        if ($hasOrderAllowDeny) { $score += 2; $matched[] = 'Order allow,deny'; }
        if ($hasPhpVariants) { $score += 3; $matched[] = 'php系拡張子列'; }
        if ($hasMixedCaseList) { $score += 4; $matched[] = '混在ケース列'; }
        if ($hasSuspectedExt) { $score += 2; $matched[] = 'suspected拡張子'; }
        if ($looksLikeSample) { $score += 3; $matched[] = '見本パターン近似'; }
    }


    // =========================================
    // 明確に安全な一般設定（誤検知防止）
    // 既存ロジックに影響を与えない早期除外
    // =========================================

    // AddType .wpress（All-in-One WP Migration）
    $isWpress = (strpos($lc, 'addtype application/octet-stream .wpress') !== false);

    // DirectoryIndex index.php
    $isDirectoryIndex = (strpos($lc, 'directoryindex index.php') !== false);

    // Options -Indexes
    $isNoIndexes = (strpos($lc, 'options -indexes') !== false);

    // IfModule ブロックのみで構成されているか
    $onlyIfModule = (
        preg_match('/^\s*(<ifmodule[\s\S]+<\/ifmodule>\s*)+$/i', trim($content)) === 1
    );

    // 明確に通常用途のみの場合は low で除外
    if ($onlyIfModule && ($isWpress || $isDirectoryIndex || $isNoIndexes)) {
        $res['is_bad'] = false;
        $res['severity'] = 'low';
        $res['kind'] = 'safe_common';
        $res['reasons'][] = '一般的な WordPress / プラグイン由来の設定のみで構成されています。';
        return $res;
    }


    // 判定
    // - 14以上: high
    // - 10以上: medium（削除候補）
    if ($score >= 14) {
        $res['is_bad'] = true;
        $res['severity'] = 'high';
        $res['kind'] = $phpEnable ? 'php_enable' : 'redirect';
        if ($phpEnable) $res['reasons'][] = 'PHP実行の許可/強制に該当するルールが含まれています。';
        if ($hasRewriteToExternal) $res['reasons'][] = '外部URLへの誘導・参照が含まれています。';
    } elseif ($score >= 10) {
        $res['is_bad'] = true;
        $res['severity'] = 'medium';
        $res['kind'] = 'deny_cloak';
        $res['reasons'][] = 'FilesMatchで対象拡張子を指定し、Denyでアクセスを遮断する構造が見られます（隠蔽目的の可能性）。';
    } else {
        $res['is_bad'] = false;
        $res['severity'] = 'low';
        $res['kind'] = 'unknown';
        $res['reasons'][] = '不正候補の閾値に満たないため除外しました。';
    }

    $res['score'] = $score;
    $res['matched'] = array_values(array_unique($matched));
    return $res;
}

/* =============================
 * ① 555ディレクトリ配下のファイル列挙
 * ============================= */
function list_files_under_for_555(string $dir, int $maxDepth, string $root): array {
    $files = [];

    // このツール自身を除外（ファイル名が変わっても realpath で一致させる）
    $selfReal = realpath(__FILE__);

    $it = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, FilesystemIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );

    foreach ($it as $f) {
        if ($it->getDepth() >= $maxDepth) continue;
        if (!$f->isFile() && !$f->isLink()) continue;

        $abs = norm($f->getPathname());

        // 追加: 自分自身は表示しない
        $absReal = realpath($abs);
        if ($selfReal !== false && $absReal !== false && $absReal === $selfReal) {
            continue;
        }

        $files[] = [
            'rel' => rel_path($abs, $root),
            'perm' => substr(sprintf('%o', $f->getPerms()), -3),
            'ext' => strtolower($f->getExtension()),
            'is_php' => strtolower($f->getExtension()) === 'php',
        ];
    }

    usort($files, fn($a, $b) => strcmp($a['rel'], $b['rel']));
    return $files;
}


/* =============================
 * ⑥ public_html 外の PHP / 権限555 / 不正 .htaccess（確認専用）
 * - PHPはディレクトリ単位で再帰検出
 * - 555 + PHP 同居は高危険
 * - 表示は2階層まで
 * ============================= */
function scan_outside_public_html(string $publicRoot): array
{
    $result = [];

    $parent = realpath(dirname($publicRoot));
    if ($parent === false) return [];

    $denyRoots = ['/', '/bin', '/usr', '/etc', '/var', '/proc', '/sys'];


    foreach (new DirectoryIterator($parent) as $base) {
        if ($base->isDot()) continue;

        // 追加: ディレクトリ以外（.htaccess等のファイル）は除外
        if (!$base->isDir()) continue;

        $baseAbs = realpath($base->getPathname());
        if (!$baseAbs || $baseAbs === $publicRoot) continue;

        // 追加: 念のため（シンボリックリンク等でズレるケース対策）
        if (!is_dir($baseAbs)) continue;

        $denyRoots = ['/', '/bin', '/usr', '/etc', '/var', '/proc', '/sys'];

        foreach ($denyRoots as $deny) {
            if ($baseAbs === $deny || strpos($baseAbs . '/', $deny . '/') === 0) {
                continue 2;
            }
        }
        $phpDirs = [];       // PHPが存在するディレクトリ
        $perm555Dirs = [];   // 555ディレクトリ
        $badHtaccessDirs = [];

        // 再帰精査
        $it = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($baseAbs, FilesystemIterator::SKIP_DOTS),
            RecursiveIteratorIterator::SELF_FIRST
        );

        foreach ($it as $f) {
            $path = $f->getPathname();

            // PHP検出（ディレクトリ単位）
            if ($f->isFile() && strtolower($f->getExtension()) === 'php') {
                $phpDirs[dirname($path)] = true;
            }

            // 555ディレクトリ
            if ($f->isDir()) {
                $perm = substr(sprintf('%o', $f->getPerms()), -3);
                if ($perm === '555') {
                    $perm555Dirs[$path] = true;
                }
            }

            // 不正 .htaccess
            if ($f->isFile() && $f->getFilename() === '.htaccess') {
                $ht = is_suspicious_htaccess(
                    $path,
                    $f->getFilename(),
                    $publicRoot
                );
                if ($ht['is_bad']) {
                    $badHtaccessDirs[dirname($path)] = true;
                }
            }
        }

        // 555ディレクトリ表示（PHP同居を判定）
        foreach ($perm555Dirs as $dirAbs => $_) {

            // ★ realpath を使わない（555対策）
            $rel = ltrim(str_replace($parent, '', $dirAbs), '/');

            $hasPhp = false;
            foreach ($phpDirs as $phpDir => $_p) {
                if (strpos($phpDir . '/', $dirAbs . '/') === 0) {
                    $hasPhp = true;
                    break;
                }
            }

            $result[] = [
                'path'   => $rel,
                'type'   => 'perm',
                'perm'   => '555',
                'detail' => $hasPhp
                    ? 'ディレクトリ権限 555 + PHP 同居（高危険）'
                    : 'ディレクトリ権限 555',
                'danger' => $hasPhp,
            ];
        }


        // 不正 .htaccess 表示
        foreach ($badHtaccessDirs as $dirAbs => $_) {
            $rel = ltrim(str_replace($parent, '', $dirAbs), '/');
            if (substr_count($rel, '/') > 1) continue;

            $perm = substr(sprintf('%o', fileperms($dirAbs)), -3);

            $result[] = [
                'path'   => $rel,
                'type'   => 'htaccess',
                'perm'   => $perm,
                'detail' => '不正 .htaccess 検出',
            ];
        }
    }

    return $result;
}


/**
 * ④ マルウェア判定（rot13ローダー型：確定級）
 * - rot13でURLを隠し、外部からコードを取得してevalで実行するタイプを検知
 * - 誤検知を避けるため、複数の強い特徴の組み合わせでのみ true にする
 */
function is_definitely_malware_rot13_loader(string $filePath, int $maxReadBytes = 450000): array
{
    $res = [
        'is_malware' => false,
        'score'      => 0,
        'matched'    => [],
        'reasons'    => [],
    ];

    // ファイルチェック
    if (!is_file($filePath) || !is_readable($filePath)) {
        $res['reasons'][] = 'ファイルを読めません';
        return $res;
    }

    // 内容取得（先頭Nバイト）
    $content = read_file_head($filePath, $maxReadBytes);
    if ($content === '') {
        $res['reasons'][] = '内容を取得できません';
        return $res;
    }

    $lc      = strtolower($content);
    $score   = 0;
    $matched = [];

    // eval
    $hasEval = (strpos($lc, 'eval(') !== false);
    $hasEvalPhpTagConcat = (bool)preg_match('/eval\s*\(\s*["\']\s*<\?\>\s*["\']\s*\.\s*\$/i', $content);
    if ($hasEval) {
        $score += 6;
        $matched[] = 'eval(';
    }
    if ($hasEvalPhpTagConcat) {
        $score += 10;
        $matched[] = 'eval("?>" . $var)';
    }

    // rot13 + uggcf:// (rot13で http(s):// を隠す典型)
    $hasRot13 = (strpos($lc, 'str_rot13(') !== false);
    $hasUggcf = (strpos($lc, 'uggcf://') !== false) || (strpos($lc, 'uggcf%3a%2f%2f') !== false);
    if ($hasRot13) {
        $score += 4;
        $matched[] = 'str_rot13(';
    }
    if ($hasUggcf) {
        $score += 10;
        $matched[] = 'uggcf://';
    }

    // 外部取得手段（curl / file_get_contents / fopen+stream_get_contents）
    $hasCurlInit        = (strpos($lc, 'curl_init(') !== false);
    $hasCurlExec        = (strpos($lc, 'curl_exec(') !== false);
    $hasFileGetContents = (strpos($lc, 'file_get_contents(') !== false);
    $hasFopenStream     = (strpos($lc, 'fopen(') !== false) && (strpos($lc, 'stream_get_contents(') !== false);

    $fetchCount = 0;
    if ($hasCurlInit && $hasCurlExec) $fetchCount++;
    if ($hasFileGetContents) $fetchCount++;
    if ($hasFopenStream) $fetchCount++;

    if ($fetchCount >= 2) {
        $score += 6;
        $matched[] = '外部取得フォールバック';
    } elseif ($fetchCount === 1) {
        $score += 1;
        $matched[] = '外部取得';
    }

    // SSL検証無効化（curl_setopt(..., CURLOPT_SSL_VERIFYPEER, 0) 等）
    $hasSslOff1 = (strpos($lc, 'curlopt_ssl_verifypeer') !== false)
        && (preg_match('/curl_setopt\s*\(\s*\$\w+\s*,\s*curlopt_ssl_verifypeer\s*,\s*0\s*\)/i', $content) === 1);

    $hasSslOff2 = (strpos($lc, 'curlopt_ssl_verifyhost') !== false)
        && (preg_match('/curl_setopt\s*\(\s*\$\w+\s*,\s*curlopt_ssl_verifyhost\s*,\s*0\s*\)/i', $content) === 1);

    if ($hasSslOff1 || $hasSslOff2) {
        $score += 4;
        $matched[] = 'SSL検証無効化';
    }

    // doactスイッチ（REQUEST/SESSIONに doact を保持して挙動切替）
    $hasSessionStart = (strpos($lc, 'session_start') !== false);

    $hasRequestDoact =
        (strpos($lc, '$_request') !== false) &&
        (
            (strpos($lc, '"doact"') !== false) ||
            (preg_match('/\$_request\s*\[\s*["\']\\\\144\\\\157\\\\141\\\\143\\\\x74["\']\s*\]/i', $content) === 1) ||
            (preg_match('/\$_request\s*\[\s*["\']\\\\x64\\\\x6f\\\\x61\\\\x63\\\\x74["\']\s*\]/i', $content) === 1)
        );

    $hasSessionDoact =
        (strpos($lc, '$_session') !== false) &&
        (
            (strpos($lc, '"doact"') !== false) ||
            (preg_match('/\$_session\s*\[\s*["\']doact["\']\s*\]/i', $content) === 1) ||
            (preg_match('/\$_session\s*\[\s*["\']\\\\x64\\\\x6f\\\\x61\\\\x63\\\\x74["\']\s*\]/i', $content) === 1)
        );

    if ($hasSessionStart && $hasRequestDoact && $hasSessionDoact) {
        $score += 7;
        $matched[] = 'doactスイッチ';
    }

    // goto難読化（多用される）
    $m1 = [];
    $gotoCount = preg_match_all('/\bgoto\b/i', $content, $m1);
    if (is_int($gotoCount) && $gotoCount >= 4) {
        $score += 4;
        $matched[] = 'goto難読化';
    }

    // 確定判定ロジック
    $hasCoreCombo   = $hasEval && $hasRot13 && $hasUggcf && ($fetchCount >= 1);
    $hasStrongCombo = $hasEvalPhpTagConcat && $hasUggcf && ($fetchCount >= 2);

    if ($hasStrongCombo || ($hasCoreCombo && $score >= 18)) {
        $res['is_malware'] = true;
        $res['reasons'][]  = 'rot13でURLを隠し、外部からコードを取得してevalで実行するローダー型の特徴が揃っています。';
        if ($hasSessionStart && $hasRequestDoact && $hasSessionDoact) {
            $res['reasons'][] = 'doactパラメータをセッションに保存して挙動を切り替える構造です。';
        }
    } else {
        $res['is_malware'] = false;
        $res['reasons'][]  = '確定級の組み合わせに満たないため未確定扱いです。';
    }

    $res['score']   = $score;
    $res['matched'] = array_values(array_unique($matched));

    return $res;
}


/**
 * WebShell / C2管理型バックドアの確定検知
 * - goto多用・難読化
 * - 外部C2通信
 * - exec / shell_exec 等のOS実行
 * - base64_decode 多用
 * - SSL検証無効 curl
 * - セッション鍵による制御
 */
function is_definitely_malware_webshell(string $filePath, int $maxReadBytes = 500000): array
{
    $res = [
        'is_malware' => false,
        'score'      => 0,
        'matched'    => [],
        'reasons'    => [],
    ];

    if (!is_file($filePath) || !is_readable($filePath)) {
        $res['reasons'][] = 'ファイルを読めません';
        return $res;
    }

    $content = read_file_head($filePath, $maxReadBytes);
    if ($content === '') {
        $res['reasons'][] = '内容を取得できません';
        return $res;
    }

    $lc = strtolower($content);
    $score = 0;
    $matched = [];

    /* =====================================================
     * 1. 制御フロー難読化（goto多用）
     * ===================================================== */
    $gotoCount = preg_match_all('/\bgoto\b/i', $content, $m);
    if ($gotoCount >= 4) {
        $score += 6;
        $matched[] = "goto多用({$gotoCount})";
    }

    /* =====================================================
     * 2. base64_decode 多用（WebShell典型）
     * ===================================================== */
    $b64Count = preg_match_all('/base64_decode\s*\(/i', $content, $m);
    if ($b64Count >= 2) {
        $score += 6;
        $matched[] = "base64_decode多用({$b64Count})";
    } elseif ($b64Count === 1) {
        $score += 2;
        $matched[] = "base64_decode";
    }

    /* =====================================================
     * 3. OSコマンド実行系（致命）
     * ===================================================== */
    if (preg_match('/\b(exec|shell_exec|system|passthru|proc_open)\s*\(/i', $content)) {
        $score += 10;
        $matched[] = 'OSコマンド実行';
    }

    /* =====================================================
     * 4. 外部C2通信（curl / file_get_contents）
     * ===================================================== */
    $hasCurl = (strpos($lc, 'curl_init(') !== false) && (strpos($lc, 'curl_exec(') !== false);
    $hasFgc  = (strpos($lc, 'file_get_contents(') !== false);
    $hasFopen = (strpos($lc, 'fopen(') !== false && strpos($lc, 'stream_get_contents(') !== false);

    $fetchCount = 0;
    if ($hasCurl) $fetchCount++;
    if ($hasFgc)  $fetchCount++;
    if ($hasFopen) $fetchCount++;

    if ($fetchCount >= 2) {
        $score += 6;
        $matched[] = '外部通信フォールバック';
    } elseif ($fetchCount === 1) {
        $score += 3;
        $matched[] = '外部通信';
    }

    /* =====================================================
     * 5. SSL検証無効化（C2通信の常套）
     * ===================================================== */
    if (
        preg_match('/CURLOPT_SSL_VERIFYPEER\s*,\s*(0|false)/i', $content) ||
        preg_match('/CURLOPT_SSL_VERIFYHOST\s*,\s*0/i', $content)
    ) {
        $score += 4;
        $matched[] = 'SSL検証無効化';
    }

    /* =====================================================
     * 6. セッションベース制御（簡易認証）
     * ===================================================== */
    $hasSessionStart = (strpos($lc, 'session_start') !== false);
    $hasSessionKey   = (preg_match('/\$_session\s*\[\s*[\'"][a-z0-9_]{4,20}[\'"]\s*\]/i', $content) === 1);
    $hasRequestKey   = (preg_match('/\$_(get|post|request)\s*\[\s*[\'"][a-z0-9_]{4,20}[\'"]\s*\]/i', $content) === 1);

    if ($hasSessionStart && $hasSessionKey && $hasRequestKey) {
        $score += 6;
        $matched[] = 'セッション鍵制御';
    }

    /* =====================================================
     * 7. WordPress無関係ファイルでの存在（決定打）
     * ===================================================== */
    if (
        strpos($filePath, '/uploads/') !== false ||
        strpos($filePath, '/cache/') !== false ||
        strpos($filePath, '/tmp/') !== false
    ) {
        $score += 4;
        $matched[] = '不正配置ディレクトリ';
    }

    /* =====================================================
     * 確定条件
     * ===================================================== */
    if (
        $score >= 22 ||
        (
            $hasCurl &&
            preg_match('/\b(exec|shell_exec|system|passthru)\s*\(/i', $content) &&
            $gotoCount >= 4
        )
    ) {
        $res['is_malware'] = true;
        $res['reasons'][] =
            'WebShell/C2管理型バックドアの典型的な特徴（OS実行・外部通信・難読化）が複合的に検出されました。';
    } else {
        $res['reasons'][] = 'WebShell特徴はあるが確定条件未満です。';
    }

    $res['score']   = $score;
    $res['matched'] = array_values(array_unique($matched));

    return $res;
}




/* =============================
 * スキャン本体
 * ============================= */
function scan_site(
    string $root,
    ?string $uploadsAbs,
    array $wpRootStandardPhp,
    array $suspectNamesExact
): array {
    $result = [
        'perm555' => [],
        'permAdvice' => [],
        'tampered' => [],
        'malware' => [],
        'htaccess' => [],
        'counts' => [
            'scanned_files' => 0,
            'scanned_php' => 0,
            'scanned_htaccess' => 0,
        ],
        'outside' => [],
    ];

    
    // このツール自身を除外（ファイル名が変わっても realpath で一致させる）
    $selfReal = realpath(__FILE__);

    $uploadsN = $uploadsAbs ? rtrim(norm($uploadsAbs), '/') . '/' : null;

    $adviceMap = [];
    $perm555DirsAbs = [];

    $it = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($root, FilesystemIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );

    foreach ($it as $info) {
        /** @var SplFileInfo $info */
        $abs = norm($info->getPathname());

        // 自分自身はスキャン対象から外す（どのカテゴリにも出さない）
        $absReal = realpath($abs);
        if ($selfReal !== false && $absReal !== false && $absReal === $selfReal) {
            continue;
        }

        // uploads は基本スキップ（ただし .htaccess は走査）
        // 例外: uploads 配下でも "ディレクトリ" の 555 は ① に拾う
        $inUploads = ($uploadsN && strpos($abs . '/', $uploadsN) === 0);
        if ($inUploads) {
            if ($info->isDir()) {
                $perm = substr(sprintf('%o', $info->getPerms()), -3);
                if ($perm === '555') {
                    $perm555DirsAbs[] = $abs;
                }
                continue;
            }

            $isHtaccess = ($info->isFile() && $info->getFilename() === '.htaccess');
            if (!$isHtaccess) {
                continue;
            }
        }



        $rel = rel_path($abs, $root);

        if ($info->isDir()) {
            $perm = substr(sprintf('%o', $info->getPerms()), -3);

            if ($perm === '555') {
                $perm555DirsAbs[] = $abs;
            }

            $d = depth_of_rel($rel);
            if ($d >= 1 && $d <= 2) {
                if (!isset($adviceMap[$rel])) {
                    $adviceMap[$rel] = [
                        'path' => $rel,
                        'perm' => $perm,
                        'readable' => is_readable($abs) ? 'yes' : 'no',
                        'writable' => is_writable($abs) ? 'yes' : 'no',
                        'total_files' => 0,
                        'bad_files_total' => 0,
                        'bad_perm_counts' => [],
                    ];
                } else {
                    $adviceMap[$rel]['perm'] = $perm;
                    $adviceMap[$rel]['readable'] = is_readable($abs) ? 'yes' : 'no';
                    $adviceMap[$rel]['writable'] = is_writable($abs) ? 'yes' : 'no';
                }
            }

            continue;
        }

        if (!$info->isFile() && !$info->isLink()) continue;
        $result['counts']['scanned_files']++;

        $perm = substr(sprintf('%o', $info->getPerms()), -3);

        $bucket = bucket_key($rel, 2);
        if ($bucket !== '.' && isset($adviceMap[$bucket])) {
            $adviceMap[$bucket]['total_files']++;

            if ($perm !== RECOMMENDED_FILE_PERM) {
                $adviceMap[$bucket]['bad_files_total']++;
                if (!isset($adviceMap[$bucket]['bad_perm_counts'][$perm])) {
                    $adviceMap[$bucket]['bad_perm_counts'][$perm] = 0;
                }
                $adviceMap[$bucket]['bad_perm_counts'][$perm]++;
            }
        }

        // ⑤ .htaccess
        if ($info->getFilename() === '.htaccess') {
            $result['counts']['scanned_htaccess']++;

            $absReal = realpath($abs);
            if ($absReal !== false) {
                $ht = is_suspicious_htaccess($absReal, $rel, $root);
                if ($ht['is_bad']) {
                    $result['htaccess'][] = [
                        'rel' => $rel,
                        'perm' => $perm,
                        'severity' => (string)$ht['severity'],
                        'kind' => (string)$ht['kind'],
                        'score' => (string)$ht['score'],
                        'matched' => implode(', ', $ht['matched']),
                        'reason' => implode(' / ', $ht['reasons']),
                    ];
                }
            }
        }

        // PHPの検知
        $ext = strtolower(pathinfo($rel, PATHINFO_EXTENSION));
        if ($ext !== 'php') continue;

        $result['counts']['scanned_php']++;

        $base = basename($rel);
        $isRootLevel = (strpos($rel, '/') === false);

        $isSuspectName = in_array(strtolower($base), array_map('strtolower', $suspectNamesExact), true);

        if ($isRootLevel && !in_array($base, $wpRootStandardPhp, true)) {
            $result['tampered'][] = [
                'rel' => $rel,
                'perm' => $perm,
                'reason' => '直下にあるWP標準外のPHP',
            ];
        } elseif ($isSuspectName) {
            $result['tampered'][] = [
                'rel' => $rel,
                'perm' => $perm,
                'reason' => '怪しいファイル名（タイポ狙い）',
            ];
        }

        $absReal = realpath($abs);
        if ($absReal !== false) {
            $g = is_definitely_malware_gecko_shell($absReal);
            if ($g['is_malware']) {
                $result['malware'][] = [
                    'rel' => $rel,
                    'perm' => $perm,
                    'kind' => 'gecko',
                    'score' => $g['score'],
                    'matched' => implode(', ', $g['matched']),
                    'reason' => implode(' / ', $g['reasons']),
                ];
                continue;
            }

            $r = is_definitely_malware_rot13_loader($absReal);
            if ($r['is_malware']) {
                $result['malware'][] = [
                    'rel' => $rel,
                    'perm' => $perm,
                    'kind' => 'rot13_loader',
                    'score' => $r['score'],
                    'matched' => implode(', ', $r['matched']),
                    'reason' => implode(' / ', $r['reasons']),
                ];
                continue;
            }


            $w = is_definitely_malware_webshell($absReal);
            if ($w['is_malware']) {
                $result['malware'][] = [
                    'rel'     => $rel,
                    'perm'    => $perm,
                    'kind'    => 'webshell',
                    'score'   => $w['score'],
                    'matched' => implode(', ', $w['matched']),
                    'reason'  => implode(' / ', $w['reasons']),
                ];
                continue;
            }

        }
    }

    foreach ($perm555DirsAbs as $dAbs) {
        $dRel = rel_path($dAbs, $root);
        $files = list_files_under_for_555($dAbs, MAX_FILE_DEPTH_IN_555, $root);

        $result['perm555'][] = [
            'dir_rel' => $dRel,
            'dir_perm' => file_perm_octal($dAbs),
            'files' => $files,
        ];
    }

    $permAdvice = [];
    foreach ($adviceMap as $dirRel => $st) {
        $dirBad = ($st['perm'] !== RECOMMENDED_DIR_PERM);
        $deny = ($st['readable'] !== 'yes') || ($st['writable'] !== 'yes');
        $hasBadFiles = ($st['bad_files_total'] > 0);

        if ($deny || $dirBad || $hasBadFiles) {
            $breakdown = '';
            if (!empty($st['bad_perm_counts'])) {
                ksort($st['bad_perm_counts']);
                $parts = [];
                foreach ($st['bad_perm_counts'] as $p => $cnt) {
                    $parts[] = $p . ':' . $cnt;
                }
                $breakdown = implode(', ', $parts);
            }

            $permAdvice[] = [
                'path' => $st['path'],
                'dir_perm' => $st['perm'],
                'readable' => $st['readable'],
                'writable' => $st['writable'],
                'total_files' => (string)$st['total_files'],
                'bad_files_total' => (string)$st['bad_files_total'],
                'bad_breakdown' => $breakdown === '' ? '-' : $breakdown,
            ];
        }
    }

    usort($permAdvice, fn($a, $b) => strcmp($a['path'], $b['path']));
    $result['permAdvice'] = $permAdvice;

    usort($result['tampered'], fn($a, $b) => strcmp($a['rel'], $b['rel']));
    usort($result['malware'], fn($a, $b) => strcmp($a['rel'], $b['rel']));
    usort($result['htaccess'], fn($a, $b) => strcmp($a['rel'], $b['rel']));

    $result['outside'] = scan_outside_public_html($root);

    return $result;
}

/* =============================
 * アクセス制御（秘密キー必須）
 * ============================= */
if (!key_ok()) {
    header('Content-Type: text/html; charset=UTF-8');
    echo '<h2>アクセス不可</h2><p>秘密キーが必要です</p>';
    exit;
}

// ツールは /public_html/scan/ に置く想定。
// スキャン対象は public_html（= scan の1つ上）に固定する。
$ROOT = realpath(__DIR__ . DIRECTORY_SEPARATOR . '..') ?: (realpath(__DIR__) ?: __DIR__);
$UPLOADS_ABS = realpath($ROOT . DIRECTORY_SEPARATOR . UPLOADS_REL);
$csrf = make_csrf_token();

$mode = isset($_GET['mode']) ? (string)$_GET['mode'] : 'scan';
if ($mode === 'run' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    $mode = 'scan';
}
// selected の取得（confirmではPOST → session保存、runではsessionから取得）
$selected = [];


if ($mode === 'stash' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $selected = isset($_POST['selected']) && is_array($_POST['selected']) ? $_POST['selected'] : [];

    $selected = array_values(
        array_unique(
            array_map(
                fn($p) => ltrim(norm((string)$p), '/'),
                array_filter($selected, fn($x) => $x !== '')
            )
        )
    );

    $_SESSION['wp_scan_fix_selected'] = $selected;

    header('Content-Type: application/json; charset=UTF-8');
    echo json_encode(['ok' => true, 'count' => count($selected)]);
    exit;
}



if ($mode === 'confirm' && $_SERVER['REQUEST_METHOD'] === 'POST') {
    // confirm 直前：POSTから受けて session に保存
    $selected = isset($_POST['selected']) && is_array($_POST['selected']) ? $_POST['selected'] : [];
    $selected = array_values(
        array_filter(
            array_map('strval', $selected),
            fn($x) => $x !== ''
        )
    );
    $selected = array_map(fn($p) => ltrim(norm((string)$p), '/'), $selected);
    $selected = array_values(array_unique($selected));

    $_SESSION['wp_scan_fix_selected'] = $selected;

} elseif ($mode === 'run') {
    // run：session から取得（POSTは使わない）
    $selected = isset($_SESSION['wp_scan_fix_selected']) && is_array($_SESSION['wp_scan_fix_selected'])
        ? $_SESSION['wp_scan_fix_selected']
        : [];
}

$requestedAction = isset($_POST['action_kind']) ? (string)$_POST['action_kind'] : 'delete';
$autoChmodForDelete = isset($_POST['auto_chmod']) ? parse_bool($_POST['auto_chmod']) : true;

$runLog = [];
$runOk = false;

$scan = scan_site($ROOT, $UPLOADS_ABS, $WP_ROOT_STANDARD_PHP, $SUSPECT_NAMES_EXACT);

if ($mode === 'run') {
    if (!verify_csrf_token($_POST['csrf'] ?? null)) {
        $runLog[] = ['ng', 'CSRFトークンが不正です。'];
    } elseif ($requestedAction !== 'delete') {
        $runLog[] = ['ng', '処理種別が不正です。'];
    } elseif (count($selected) === 0) {
        $runLog[] = ['ng', '選択がありません。'];
    } else {
        $allOk = true;

        $deletedDirs = [];

        $isAlreadyDeletedByParent = function (string $rel) use (&$deletedDirs): bool {
            $rel = ltrim($rel, '/');
            $relSlash = $rel . '/';
            foreach ($deletedDirs as $d) {
                if (strpos($relSlash, $d) === 0) return true;
            }
            return false;
        };

        $selfRel = rel_path(norm(__FILE__), $ROOT);


        // 親ディレクトリを先に処理する（配下ファイルの「存在しません」を減らす）
        usort($selected, function ($a, $b) {
            $da = depth_of_rel((string)$a);
            $db = depth_of_rel((string)$b);
            if ($da !== $db) return $da <=> $db; // 浅い方を先
            return strlen((string)$a) <=> strlen((string)$b); // 同階層なら短い方を先
        });

        $selfRel = rel_path(norm(__FILE__), $ROOT);


        $processed = [];

        foreach ($selected as $rel) {
            $rel = ltrim($rel, '/');

            if ($rel === '.' || $rel === '') {
                $runLog[] = ['ng', 'ルート(.)は削除できません。'];
                $allOk = false;
                continue;
            }
            if ($rel === $selfRel) {
                $runLog[] = ['ng', 'このツール自身は削除対象にできません。'];
                $allOk = false;
                continue;
            }

            // 重複選択はOKでスキップ
            if (isset($processed[$rel])) {
                $runLog[] = ['ok', "スキップ（重複選択）: {$rel}"];
                continue;
            }
            $processed[$rel] = true;

            // 親ディレクトリ削除済みなら、子はOKでスキップ
            if ($isAlreadyDeletedByParent($rel)) {
                $runLog[] = ['ok', "削除済（親ディレクトリに含まれる）: {$rel}"];
                continue;
            }

            $targetAbs = $ROOT . DIRECTORY_SEPARATOR . $rel;
            $targetReal = realpath($targetAbs);

            // ここが重要: 存在しないは基本OK扱い（親削除済み/先に削除済みが多い）
            if ($targetReal === false) {
                $runLog[] = ['ok', "存在しません（既に削除済の可能性）: {$rel}"];
                continue;
            }

            if (!is_within_root($targetReal, $ROOT)) {
                $runLog[] = ['ng', "ROOT外のため拒否: {$rel}"];
                $allOk = false;
                continue;
            }

            // 削除前に種別を確定
            $wasDir = is_dir($targetReal);

            if ($autoChmodForDelete) {
                if ($wasDir) {
                    @chmod($targetReal, 0755);
                    chmod_recursive($targetReal, 0755, 0644, 25, 0);
                } elseif (is_file($targetReal) || is_link($targetReal)) {
                    @chmod($targetReal, 0644);
                }
            }

            $ok = delete_path_recursive($targetReal);
            if ($ok) {
                // ここが重要: 削除前にdir判定した $wasDir を使う
                if ($wasDir) {
                    $deletedDirs[] = rtrim($rel, '/') . '/';
                }
                $runLog[] = ['ok', "削除: {$rel}"];
            } else {
                $runLog[] = ['ng', "削除失敗: {$rel}（権限/所有者/ロックの可能性）"];
                $allOk = false;
            }
        }



        $runOk = $allOk;
        unset($_SESSION['wp_scan_fix_selected']);

    }

    $mode = 'scan';
}




/* =============================
 * HTML
 * ============================= */
$key = current_key();
$actionConfirm = '?key=' . rawurlencode($key) . '&mode=confirm';
$actionRun = '?key=' . rawurlencode($key) . '&mode=run';
$actionScan = '?key=' . rawurlencode($key) . '&mode=scan';

?>
<!DOCTYPE html>
<html lang="ja">
<head>
<meta charset="UTF-8">
<meta name="robots" content="noindex,nofollow">
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>WP Scan Fix <?php echo h(TOOL_VERSION); ?></title>
<style>
    body { font-family: system-ui, -apple-system, Segoe UI, Roboto, "Hiragino Kaku Gothic ProN", "Noto Sans JP", "Yu Gothic", sans-serif; margin: 16px; }
    h1 { font-size: 20px; margin: 0 0 12px; }
    h2 { margin-top: 28px; font-size: 18px; }
    .box { border: 1px solid #333; padding: 12px; margin-bottom: 14px; }
    .muted { color: #555; }
    .danger { color: #b00020; font-weight: 700; }
    .ok { color: #006400; font-weight: 700; }
    .btn { padding: 8px 12px; border: 1px solid #333; background: #f6f6f6; cursor: pointer; }
    .btn-primary { background: #e8f0ff; }
    .btn-danger { background: #ffe8e8; }
    table { border-collapse: collapse; width: 100%; }
    th, td { border: 1px solid #333; padding: 6px 8px; vertical-align: top; }
    th { background: #f3f3f3; }
    .small { font-size: 12px; }
    .nowrap { white-space: nowrap; }
    .group { background: #fafafa; font-weight: 700; }
    .row-actions { display: flex; gap: 8px; align-items: center; flex-wrap: wrap; }
    .pill { display:inline-block; padding:2px 6px; border:1px solid #333; border-radius: 999px; font-size: 12px; }
    .mono { font-family: ui-monospace, SFMono-Regular, Menlo, Monaco, Consolas, "Liberation Mono", "Courier New", monospace; }
    .actionbar {
        position: sticky;
        top: 0;
        z-index: 9999;
        background: #fff;
        border: 2px solid #333;
        padding: 10px 12px;
        margin-bottom: 14px;
    }
    tr.danger {
        background: #ffe6e6;
    }
    tr.danger td {
        color: #a40000;
        font-weight: bold;
    }

</style>
</head>
<body>

<div class="box">
    <h1>WP Scan Fix（実行版） <span class="muted">ver <?php echo h(TOOL_VERSION); ?></span></h1>
    <div class="muted small">
        このページは秘密キー必須です。表示 -> チェック -> 確認 -> 削除 の2段階で実行されます。
        uploads は基本スキップします（.htaccessのみ例外で走査）。
    </div>
    <div class="muted small">
        スキャン対象ルート: <span class="mono">ここに実行パスが表示される</span>
    </div>
    <div class="muted small">
        スキャン統計:
        files <?php echo h((string)$scan['counts']['scanned_files']); ?> /
        php <?php echo h((string)$scan['counts']['scanned_php']); ?> /
        htaccess <?php echo h((string)$scan['counts']['scanned_htaccess']); ?>
    </div>
</div>

<div class="actionbar">
    <div class="row-actions">
        <a class="btn" href="<?php echo h($actionScan); ?>">再スキャン</a>
        <button type="button" class="btn" id="btnAllOnTop">全て選択</button>
        <button type="button" class="btn" id="btnAllOffTop">全て解除</button>
        <button type="button" class="btn btn-primary" id="btnConfirmTop">確認画面へ</button>
    </div>

    <div class="muted small" style="margin-top:6px;">
        まず一覧でチェック → 「確認画面へ」 → 確認画面の「最終削除」で実行
    </div>
</div>



<?php if (!empty($runLog)): ?>
<div class="box">
    <div><strong>実行ログ</strong> <?php echo $runOk ? '<span class="ok">完了</span>' : '<span class="danger">一部失敗</span>'; ?></div>
    <div class="row-actions" style="margin-top:10px;">
        <a class="btn" href="<?php echo h($actionScan); ?>">再スキャン</a>
    </div>
    <table>
        <tr><th style="width:80px;">結果</th><th>内容</th></tr>
        <?php foreach ($runLog as $l): ?>
        <tr>
            <td class="<?php echo $l[0] === 'ok' ? 'ok' : 'danger'; ?>"><?php echo h($l[0]); ?></td>
            <td><?php echo h($l[1]); ?></td>
        </tr>
        <?php endforeach; ?>
    </table>
    <div style="margin-top:10px;">
        <a class="btn" href="<?php echo h($actionScan); ?>">再スキャン</a>
    </div>
</div>
<?php endif; ?>

<div class="box">
    <div><strong>削除オプション</strong></div>
    <div class="muted small">555等で削除できない場合があるため、削除前に限定深さでchmodしてから削除します。</div>
    <label>
        <input type="checkbox" name="auto_chmod" value="1" form="scanForm" <?php echo $autoChmodForDelete ? 'checked' : ''; ?>>
        削除前に chmod を実施（555対策）
    </label>
</div>

<?php

if (isset($_GET['mode']) && $_GET['mode'] === 'confirm') :
    // confirm は GET で開く（選択は stash で session 済み）ので csrf はここでは見ない
    $selected = (isset($_SESSION['wp_scan_fix_selected']) && is_array($_SESSION['wp_scan_fix_selected']))
        ? $_SESSION['wp_scan_fix_selected']
        : [];

    $okConfirm = (count($selected) > 0);
?>


<div class="box">
    <div><strong>確認画面</strong></div>

    <?php if (!$okConfirm): ?>
        <div class="danger">選択情報がありません。もう一度チェックして「確認画面へ」を押してください。</div>
    <?php else: ?>

        <table>
            <tr><th style="width:240px;">削除前 chmod</th><td><?php echo $autoChmodForDelete ? 'ON' : 'OFF'; ?></td></tr>
            <tr><th>選択数</th><td><?php echo h((string)count($selected)); ?></td></tr>
        </table>

        <?php if (count($selected) === 0): ?>
            <div class="danger" style="margin-top:10px;">選択がありません。</div>
        <?php else: ?>
            <form id="runForm" method="post" action="<?php echo h($actionRun); ?>">
                <input type="hidden" name="csrf" value="<?php echo h($csrf); ?>">
                <input type="hidden" name="action_kind" value="delete">
                <input type="hidden" name="auto_chmod" value="<?php echo $autoChmodForDelete ? '1' : '0'; ?>">

                <div class="row-actions" style="margin-top:10px;">
                    <a class="btn" href="<?php echo h($actionScan); ?>">戻る</a>
                    <button type="button" class="btn btn-danger" id="btnRun">最終削除</button>
                </div>

                

                <div class="danger" style="margin-top:10px;">まだ削除は実行されていません。</div>

                <table style="margin-top:10px;">
                    <tr>
                        <th>対象</th>
                        <th style="width:110px;">現在権限</th>
                        <th style="width:160px;">種別</th>
                    </tr>
                    <?php foreach ($selected as $p): ?>
                        <?php
                            $abs = $ROOT . DIRECTORY_SEPARATOR . ltrim((string)$p, '/');
                            $perm = file_exists($abs) ? file_perm_octal($abs) : '-';
                            $type = is_dir($abs) ? 'dir' : (is_file($abs) ? 'file' : 'other');
                        ?>
                        <tr>
                            <td class="small"><?php echo h((string)$p); ?></td>
                            <td class="nowrap"><?php echo h((string)$perm); ?></td>
                            <td><?php echo h((string)$type); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </table>

                <div class="row-actions" style="margin-top:10px;">
                    <a class="btn" href="<?php echo h($actionScan); ?>">戻る</a>
                    <button type="button" class="btn btn-danger" id="btnRun">最終削除</button>
                </div>
            </form>
        <?php endif; ?>
    <?php endif; ?>
</div>
<?php endif; ?>

<div id="fileListTop"></div>


<form id="scanForm" method="post" action="<?php echo h($actionConfirm); ?>">
    <input type="hidden" name="csrf" value="<?php echo h($csrf); ?>">
    <input type="hidden" name="action_kind" value="delete">



    <h2>① パーミッション 555 のディレクトリ（危険度：高）</h2>
    <div class="box">
        <div class="muted small">555ディレクトリは配下ファイルも列挙します。削除対象はチェックしてください。</div>

        <?php if (count($scan['perm555']) === 0): ?>
            <div class="muted">該当なし</div>
        <?php else: ?>
            <table>
                <tr>
                    <th style="width:52px;">選択</th>
                    <th>パス</th>
                    <th style="width:110px;">権限</th>
                    <th>配下ファイル（深さ <?php echo h((string)MAX_FILE_DEPTH_IN_555); ?> まで）</th>
                </tr>
                <?php foreach ($scan['perm555'] as $d): ?>
                    <?php
                        $dirRel = (string)$d['dir_rel'];
                        $bucketId = 'b1_' . md5('perm555:' . $dirRel);
                    ?>
                    <tr class="group">
                        <td class="nowrap">
                            <input type="checkbox" class="toggle-bucket" data-bucket="<?php echo h($bucketId); ?>">
                        </td>
                        <td colspan="3">
                            <?php echo h($dirRel); ?>
                            <span class="pill">dir <?php echo h((string)$d['dir_perm']); ?></span>
                            <span class="muted small">この行のチェックは「このディレクトリ配下の項目を全選択」です</span>
                        </td>
                    </tr>

                    <tr data-bucket-row="<?php echo h($bucketId); ?>">
                        <td class="nowrap">
                            <input type="checkbox" name="selected[]" value="<?php echo h($dirRel); ?>" class="chk-item" data-bucket="<?php echo h($bucketId); ?>">
                        </td>
                        <td class="small"><?php echo h($dirRel); ?></td>
                        <td class="nowrap"><?php echo h((string)$d['dir_perm']); ?></td>
                        <td class="small muted">ディレクトリ本体</td>
                    </tr>

                    <?php foreach ($d['files'] as $f): ?>
                        <?php
                            $fRel = (string)$f['rel'];
                            $note = $f['is_php'] ? 'PHPあり' : '';
                        ?>
                        <tr data-bucket-row="<?php echo h($bucketId); ?>">
                            <td class="nowrap">
                                <input type="checkbox" name="selected[]" value="<?php echo h($fRel); ?>" class="chk-item" data-bucket="<?php echo h($bucketId); ?>">
                            </td>
                            <td class="small"><?php echo h($fRel); ?></td>
                            <td class="nowrap"><?php echo h((string)$f['perm']); ?></td>
                            <td class="small"><?php echo h($note); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endforeach; ?>
            </table>
        <?php endif; ?>
    </div>

    <h2>② パーミッション/アクセス状態の注意点（確認用）</h2>
    <div class="box">
        <div class="muted small">
            ここは「危険判定」ではなく、権限・読み書き状態の変化点を拾うコーナーです。
正当な設定も混ざります。不明なものだけ確認してください（削除対象外）。推奨パーミッション
ディレクトリ: 755 / ファイル: 644
        </div>

        <?php if (count($scan['permAdvice']) === 0): ?>
            <div class="muted">該当なし</div>
        <?php else: ?>
            <table>
                <tr>
                    <th>パス（2階層まで）</th>
                    <th style="width:110px;">dir権限</th>
                    <th style="width:90px;">読込</th>
                    <th style="width:90px;">書込</th>
                    <th style="width:110px;">配下ファイル数</th>
                    <th style="width:140px;">異常ファイル数</th>
                    <th>異常内訳（権限:件数）</th>
                </tr>
                <?php foreach ($scan['permAdvice'] as $r): ?>
                    <tr>
                        <td class="small"><?php echo h($r['path']); ?></td>
                        <td class="nowrap"><?php echo h($r['dir_perm']); ?></td>
                        <td><?php echo h($r['readable']); ?></td>
                        <td><?php echo h($r['writable']); ?></td>
                        <td class="nowrap"><?php echo h($r['total_files']); ?></td>
                        <td class="nowrap"><?php echo h($r['bad_files_total']); ?></td>
                        <td class="small"><?php echo h($r['bad_breakdown']); ?></td>
                    </tr>
                <?php endforeach; ?>
            </table>
        <?php endif; ?>
        <div class="muted small" style="margin-top:8px;">
            ここは削除対象にしません（忠告のみ）。
        </div>
    </div>

    <h2>③ 改ざん PHP（削除対象）</h2>
    <div class="box">
        <div class="muted small">直下にあるWP標準外PHP、または怪しいファイル名（例: wp-confiq.php）を対象にします。</div>

        <?php if (count($scan['tampered']) === 0): ?>
            <div class="muted">該当なし</div>
        <?php else: ?>
            <table>
                <tr>
                    <th style="width:52px;">選択</th>
                    <th>パス</th>
                    <th style="width:110px;">権限</th>
                    <th>理由</th>
                </tr>
                <?php
                    $tamperedByBucket = [];
                    foreach ($scan['tampered'] as $t) {
                        $b = bucket_key((string)$t['rel'], MAX_BUCKET_DEPTH);
                        if (!isset($tamperedByBucket[$b])) $tamperedByBucket[$b] = [];
                        $tamperedByBucket[$b][] = $t;
                    }
                    ksort($tamperedByBucket);
                ?>

                <?php foreach ($tamperedByBucket as $b => $items): ?>
                    <?php $bucketId = 'b3_' . md5('tampered:' . $b); ?>
                    <tr class="group">
                        <td class="nowrap">
                            <input type="checkbox" class="toggle-bucket" data-bucket="<?php echo h($bucketId); ?>">
                        </td>
                        <td colspan="3">
                            <?php echo h($b); ?>
                            <span class="muted small">この行のチェックはこのグループを全選択</span>
                        </td>
                    </tr>

                    <?php foreach ($items as $t): ?>
                        <tr data-bucket-row="<?php echo h($bucketId); ?>">
                            <td class="nowrap">
                                <input type="checkbox" name="selected[]" value="<?php echo h((string)$t['rel']); ?>" class="chk-item" data-bucket="<?php echo h($bucketId); ?>">
                            </td>
                            <td class="small"><?php echo h((string)$t['rel']); ?></td>
                            <td class="nowrap"><?php echo h((string)$t['perm']); ?></td>
                            <td class="small"><?php echo h((string)$t['reason']); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endforeach; ?>
            </table>
        <?php endif; ?>
    </div>

    <h2>④ マルウェア PHP（削除対象）</h2>
    <div class="box">
        <div class="muted small">
            判定は確定級2パターンのみです。テーマ/プラグインの正規コード（preg_replaceやcurl_exec等）では反応しません。
        </div>

        <?php if (count($scan['malware']) === 0): ?>
            <div class="muted">該当なし</div>
        <?php else: ?>
            <table>
                <tr>
                    <th style="width:52px;">選択</th>
                    <th>パス</th>
                    <th style="width:110px;">権限</th>
                    <th style="width:140px;">種類</th>
                    <th style="width:80px;">score</th>
                    <th>一致</th>
                </tr>

                <?php
                    $malwareByBucket = [];
                    foreach ($scan['malware'] as $m) {
                        $b = bucket_key((string)$m['rel'], MAX_BUCKET_DEPTH);
                        if (!isset($malwareByBucket[$b])) $malwareByBucket[$b] = [];
                        $malwareByBucket[$b][] = $m;
                    }
                    ksort($malwareByBucket);
                ?>

                <?php foreach ($malwareByBucket as $b => $items): ?>
                    <?php $bucketId = 'b4_' . md5('malware:' . $b); ?>
                    <tr class="group">
                        <td class="nowrap">
                            <input type="checkbox" class="toggle-bucket" data-bucket="<?php echo h($bucketId); ?>">
                        </td>
                        <td colspan="5">
                            <?php echo h($b); ?>
                            <span class="muted small">この行のチェックはこのグループを全選択</span>
                        </td>
                    </tr>

                    <?php foreach ($items as $m): ?>
                        <tr data-bucket-row="<?php echo h($bucketId); ?>">
                            <td class="nowrap">
                                <input type="checkbox" name="selected[]" value="<?php echo h((string)$m['rel']); ?>" class="chk-item" data-bucket="<?php echo h($bucketId); ?>">
                            </td>
                            <td class="small"><?php echo h((string)$m['rel']); ?></td>
                            <td class="nowrap"><?php echo h((string)$m['perm']); ?></td>
                            <td class="small"><?php echo h((string)$m['kind']); ?></td>
                            <td class="nowrap"><?php echo h((string)$m['score']); ?></td>
                            <td class="small"><?php echo h((string)$m['matched']); ?></td>
                        </tr>
                        <tr data-bucket-row="<?php echo h($bucketId); ?>">
                            <td class="nowrap"></td>
                            <td colspan="5" class="small muted"><?php echo h((string)$m['reason']); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endforeach; ?>
            </table>
        <?php endif; ?>
    </div>

    <h2>⑤ 不正 .htaccess（削除対象）</h2>
    <div class="box">
        <div class="muted small">
            ルート直下のWordPress標準 .htaccess（BEGIN/END WordPress）は除外します。
        </div>

        <?php if (count($scan['htaccess']) === 0): ?>
            <div class="muted">該当なし</div>
        <?php else: ?>
            <table>
                <tr>
                    <th style="width:52px;">選択</th>
                    <th>パス</th>
                    <th style="width:110px;">権限</th>
                    <th style="width:90px;">危険度</th>
                    <th style="width:140px;">種類</th>
                    <th style="width:80px;">score</th>
                    <th>一致</th>
                </tr>

                <?php
                    $htByBucket = [];
                    foreach ($scan['htaccess'] as $m) {
                        $b = bucket_key((string)$m['rel'], MAX_BUCKET_DEPTH);
                        if (!isset($htByBucket[$b])) $htByBucket[$b] = [];
                        $htByBucket[$b][] = $m;
                    }
                    ksort($htByBucket);
                ?>

                <?php foreach ($htByBucket as $b => $items): ?>
                    <?php $bucketId = 'b5_' . md5('htaccess:' . $b); ?>
                    <tr class="group">
                        <td class="nowrap">
                            <input type="checkbox" class="toggle-bucket" data-bucket="<?php echo h($bucketId); ?>">
                        </td>
                        <td colspan="6">
                            <?php echo h($b); ?>
                            <span class="muted small">この行のチェックはこのグループを全選択</span>
                        </td>
                    </tr>

                    <?php foreach ($items as $m): ?>
                        <tr data-bucket-row="<?php echo h($bucketId); ?>">
                            <td class="nowrap">
                                <input type="checkbox" name="selected[]" value="<?php echo h((string)$m['rel']); ?>" class="chk-item" data-bucket="<?php echo h($bucketId); ?>">
                            </td>
                            <td class="small"><?php echo h((string)$m['rel']); ?></td>
                            <td class="nowrap"><?php echo h((string)$m['perm']); ?></td>
                            <td class="nowrap"><?php echo h((string)$m['severity']); ?></td>
                            <td class="small"><?php echo h((string)$m['kind']); ?></td>
                            <td class="nowrap"><?php echo h((string)$m['score']); ?></td>
                            <td class="small"><?php echo h((string)$m['matched']); ?></td>
                        </tr>
                        <tr data-bucket-row="<?php echo h($bucketId); ?>">
                            <td class="nowrap"></td>
                            <td colspan="6" class="small muted"><?php echo h((string)$m['reason']); ?></td>
                        </tr>
                    <?php endforeach; ?>
                <?php endforeach; ?>
            </table>
        <?php endif; ?>
    </div>

    <h2>⑥ public_html 外の PHP / 不正 .htaccess（確認用）</h2>
    <div class="box">
        <div class="muted small">
            public_html 外に存在する PHP ファイル、または不正判定された .htaccess を表示します。
            このセクションは確認専用で、削除・chmod・選択対象にはなりません。
        </div>

        <?php if (empty($scan['outside'])): ?>
            <div class="muted">該当なし</div>
        <?php else: ?>
            <table>
                <tr>
                    <th>パス</th>
                    <th style="width:120px;">種別</th>
                    <th style="width:90px;">権限</th>
                    <th>詳細</th>
                </tr>
                <?php foreach ($scan['outside'] as $o): ?>
                    <tr class="<?php echo !empty($o['danger']) ? 'danger' : ''; ?>">
                        <td class="small mono"><?php echo h($o['path']); ?></td>
                        <td><?php echo h($o['type']); ?></td>
                        <td class="nowrap"><?php echo h($o['perm']); ?></td>
                        <td class="small muted"><?php echo h($o['detail']); ?></td>
                    </tr>
                <?php endforeach; ?>
            </table>
        <?php endif; ?>
    </div>


    <div class="box">
        <div class="row-actions">
            <button type="button" class="btn" id="btnAllOn">全て選択</button>
            <button type="button" class="btn" id="btnAllOff">全て解除</button>
            <button type="button" class="btn btn-primary" id="btnConfirmBottom">確認画面へ</button>

        </div>
        <div class="muted small" style="margin-top:8px;">
            注意: 削除は次の確認画面で「最終削除」を押すまで実行されません。
        </div>
    </div>
</form>


<script>
(function(){
    const form = document.getElementById('scanForm');
    if (!form) return;

    const buttons = [
        document.getElementById('btnConfirmTop'),
        document.getElementById('btnConfirmBottom'),
    ].filter(Boolean);

    if (buttons.length === 0) return;

    async function handleConfirmClick() {
        const fd = new FormData();

        form.querySelectorAll('input[name="selected[]"]:checked')
            .forEach(chk => fd.append('selected[]', chk.value));

        try {
            const stashUrl = <?php echo json_encode('?key=' . rawurlencode($key) . '&mode=stash', JSON_UNESCAPED_SLASHES); ?>;
            const confirmUrl = <?php echo json_encode($actionConfirm, JSON_UNESCAPED_SLASHES); ?>;

            const res = await fetch(stashUrl, {
                method: 'POST',
                body: fd,
                credentials: 'same-origin'
            });

            if (!res.ok) throw new Error('HTTP ' + res.status);

            const j = await res.json();
            if (!j.ok) throw new Error('stash failed');

            location.href = confirmUrl;

        } catch (e) {
            alert('選択情報の保存に失敗しました');
            return;
        }
    }

    buttons.forEach(btn => {
        btn.addEventListener('click', handleConfirmClick);
    });
})();
</script>



<script>
(function(){
    const allItemChecks = () => Array.from(document.querySelectorAll('.chk-item'));
    const bucketToggles = () => Array.from(document.querySelectorAll('.toggle-bucket'));

    function setAllItems(checked){
        allItemChecks().forEach(chk => { chk.checked = checked; });
        bucketToggles().forEach(t => { t.checked = checked; });
    }

    document.getElementById('btnAllOn')?.addEventListener('click', function(){
        setAllItems(true);
    });
    document.getElementById('btnAllOff')?.addEventListener('click', function(){
        setAllItems(false);
    });

    document.getElementById('btnAllOnTop')?.addEventListener('click', function(){
        setAllItems(true);
    });
    document.getElementById('btnAllOffTop')?.addEventListener('click', function(){
        setAllItems(false);
    });


    bucketToggles().forEach(t => {
        t.addEventListener('change', function(){
            const bucketId = this.getAttribute('data-bucket');
            allItemChecks().forEach(chk => {
                if (chk.getAttribute('data-bucket') === bucketId) chk.checked = this.checked;
            });
        });
    });

    allItemChecks().forEach(chk => {
        chk.addEventListener('change', function(){
            const bucketId = this.getAttribute('data-bucket');
            const items = allItemChecks().filter(x => x.getAttribute('data-bucket') === bucketId);
            const allOn = items.length > 0 && items.every(x => x.checked);
            const toggle = document.querySelector('.toggle-bucket[data-bucket="' + bucketId + '"]');
            if (toggle) toggle.checked = allOn;
        });
    });
})();
</script>


<script>
(function(){
    const btn = document.getElementById('btnRun');
    const form = document.getElementById('runForm');
    if (!btn || !form) return;

    btn.addEventListener('click', async function () {
        btn.disabled = true;
        btn.textContent = '削除実行中…';

        const fd = new FormData(form);

        try {
            const res = await fetch(form.action, {
                method: 'POST',
                body: fd,
                credentials: 'same-origin'
            });

            if (!res.ok) throw new Error('HTTP ' + res.status);

        } catch (e) {
            alert('削除中にエラーが発生しました: ' + e.message);
            btn.disabled = false;
            btn.textContent = '最終削除';
            return;
        }

        location.href = form.action.replace('mode=run', 'mode=scan');
    });
})();

</script>

</body>
</html>
